<?php

namespace App\Traits;

use App\Services\JWTService;

/**
 * ============================================================================
 * JWT AUTH TRAIT
 * ============================================================================
 * 
 * Bu trait mevcut controller'lara eklenerek JWT sistemine geçişi sağlar.
 * 
 * KULLANIM:
 * 1. Controller'a "use JWTAuthTrait;" ekleyin
 * 2. Eski token kontrollerini kaldırın
 * 3. User ID almak için: $this->getJWTUserId()
 * 4. User bilgisi almak için: $this->getJWTUser()
 * 
 * ESKİ KOD:
 *   $token = $this->request->getGet('token');
 *   $user = $this->userModel->validateToken($token);
 *   $userId = $user['id'];
 * 
 * YENİ KOD:
 *   $userId = $this->getJWTUserId();
 *   // veya
 *   $user = $this->getJWTUser();
 * 
 * @package    Satisahazir
 * @version    2.0.0
 * ============================================================================
 */
trait JWTAuthTrait
{
    /**
     * JWT Service instance
     */
    protected ?JWTService $jwtService = null;

    /**
     * Cached JWT user data
     */
    protected ?array $jwtUser = null;

    /**
     * Cached JWT payload
     */
    protected ?array $jwtPayload = null;

    // =========================================================================
    // USER GETTERS
    // =========================================================================

    /**
     * JWT'den user ID al
     * 
     * @return int|null
     */
    protected function getJWTUserId(): ?int
    {
        $user = $this->getJWTUser();
        return $user ? (int) $user['id'] : null;
    }

    /**
     * JWT'den user bilgisini al
     * 
     * @return array|null
     */
    protected function getJWTUser(): ?array
    {
        // Önce request'ten kontrol et (filter tarafından set edilmiş olabilir)
        if ($this->request->jwt_user !== null) {
            return $this->request->jwt_user;
        }

        // Cache'den kontrol et
        if ($this->jwtUser !== null) {
            return $this->jwtUser;
        }

        // Manuel olarak token'dan çıkar
        $token = $this->extractToken();
        if (!$token) {
            return null;
        }

        try {
            $service = $this->getJWTService();
            $payload = $service->validateAccessToken($token);
            
            if ($payload && isset($payload['user'])) {
                $this->jwtUser = $payload['user'];
                $this->jwtPayload = $payload;
                return $this->jwtUser;
            }
        } catch (\Exception $e) {
            log_message('debug', 'JWT validation failed: ' . $e->getMessage());
        }

        return null;
    }

    /**
     * JWT payload al
     */
    protected function getJWTPayload(): ?array
    {
        if ($this->request->jwt_payload !== null) {
            return $this->request->jwt_payload;
        }

        if ($this->jwtPayload === null) {
            $this->getJWTUser(); // Payload'u da doldurur
        }

        return $this->jwtPayload;
    }

    /**
     * User email al
     */
    protected function getJWTUserEmail(): ?string
    {
        $user = $this->getJWTUser();
        return $user['email'] ?? null;
    }

    /**
     * User type al
     */
    protected function getJWTUserType(): ?string
    {
        $user = $this->getJWTUser();
        return $user['type'] ?? null;
    }

    // =========================================================================
    // PERMISSION CHECKS
    // =========================================================================

    /**
     * User admin mi?
     */
    protected function isAdmin(): bool
    {
        $type = $this->getJWTUserType();
        return in_array($type, ['ADMIN', 'SUPER_ADMIN']);
    }

    /**
     * User super admin mi?
     */
    protected function isSuperAdmin(): bool
    {
        return $this->getJWTUserType() === 'SUPER_ADMIN';
    }

    /**
     * User agent mi? (Agent, Admin veya Super Admin)
     */
    protected function isAgent(): bool
    {
        $type = $this->getJWTUserType();
        return in_array($type, ['AGENT', 'ADMIN', 'SUPER_ADMIN']);
    }

    /**
     * User doğrulanmış mı?
     */
    protected function isVerified(): bool
    {
        $user = $this->getJWTUser();
        return ($user['verified'] ?? false) === true;
    }

    /**
     * User giriş yapmış mı?
     */
    protected function isAuthenticated(): bool
    {
        return $this->getJWTUserId() !== null;
    }

    // =========================================================================
    // TOKEN MANAGEMENT
    // =========================================================================

    /**
     * Login response için token oluştur
     * 
     * @param array $userData User verileri (id, email, type, verified zorunlu)
     * @param array $deviceData Device verileri (device_uuid zorunlu)
     * @return array Token verileri
     */
    protected function generateLoginTokens(array $userData, array $deviceData = []): array
    {
        $service = $this->getJWTService();
        
        return $service->generateTokenPair(
            $userData,
            $deviceData['device_uuid'] ?? 'unknown',
            $deviceData['platform'] ?? 'unknown',
            $this->request->getIPAddress(),
            $this->request->getUserAgent()->getAgentString()
        );
    }

    /**
     * Logout - Token'ı revoke et
     */
    protected function revokeCurrentToken(): bool
    {
        $token = $this->extractToken();
        if (!$token) {
            return false;
        }

        $service = $this->getJWTService();
        return $service->revokeToken($token);
    }

    /**
     * Tüm token'ları revoke et (logout all)
     */
    protected function revokeAllUserTokens(): bool
    {
        $userId = $this->getJWTUserId();
        if (!$userId) {
            return false;
        }

        $service = $this->getJWTService();
        return $service->revokeAllUserTokens($userId);
    }

    /**
     * Refresh token ile yeni token al
     */
    protected function refreshTokens(string $refreshToken): ?array
    {
        $service = $this->getJWTService();
        return $service->refreshTokens($refreshToken);
    }

    // =========================================================================
    // RESPONSE HELPERS
    // =========================================================================

    /**
     * JWT hata response
     */
    protected function jwtError(string $message, string $code = 'error', int $httpCode = 400)
    {
        return $this->response
            ->setStatusCode($httpCode)
            ->setJSON([
                'code' => $code,
                'message' => $message,
            ]);
    }

    /**
     * Unauthorized response
     */
    protected function jwtUnauthorized(string $message = 'Yetkilendirme gerekli')
    {
        return $this->jwtError($message, 'unauthorized', 401);
    }

    /**
     * Forbidden response
     */
    protected function jwtForbidden(string $message = 'Bu işlem için yetkiniz yok')
    {
        return $this->jwtError($message, 'forbidden', 403);
    }

    /**
     * Login success response
     */
    protected function jwtLoginSuccess(array $userData, array $tokens, string $message = 'Giriş başarılı')
    {
        return $this->response->setJSON([
            'code' => 'success',
            'message' => $message,
            'localId' => (string) $userData['id'],
            'user_email' => $userData['email'],
            'display_name' => $userData['display_name'] ?? '',
            'type' => $userData['type'] ?? 'USER',
            'verified' => $userData['verified'] ?? false,
            'tokens' => [
                'access_token' => $tokens['access_token'],
                'refresh_token' => $tokens['refresh_token'],
                'expires_in' => $tokens['expires_in'],
                'refresh_expires_in' => $tokens['refresh_expires_in'],
                'access_expires_at' => $tokens['access_expires_at'],
                'refresh_expires_at' => $tokens['refresh_expires_at'],
            ],
        ]);
    }

    // =========================================================================
    // SECURITY LOGGING
    // =========================================================================

    /**
     * Security event logla
     */
    protected function logSecurityEvent(string $eventType, ?int $userId = null, array $data = []): void
    {
        try {
            $db = \Config\Database::connect('logs');
            $db->table('security_logs')->insert([
                'event_type' => $eventType,
                'user_id' => $userId ?? $this->getJWTUserId(),
                'ip_address' => $this->request->getIPAddress(),
                'user_agent' => $this->request->getUserAgent()->getAgentString(),
                'data' => json_encode($data),
                'created_at' => date('Y-m-d H:i:s'),
            ]);
        } catch (\Exception $e) {
            log_message('error', 'Security log error: ' . $e->getMessage());
        }
    }

    // =========================================================================
    // PRIVATE HELPERS
    // =========================================================================

    /**
     * Request'ten token çıkar
     */
    private function extractToken(): ?string
    {
        // 1. Request'te zaten varsa (filter tarafından set edilmiş)
        if (!empty($this->request->jwt_token)) {
            return $this->request->jwt_token;
        }

        // 2. Authorization header
        $authHeader = $this->request->getHeaderLine('Authorization');
        if (!empty($authHeader) && preg_match('/Bearer\s+(.+)$/i', $authHeader, $matches)) {
            return $matches[1];
        }

        // 3. X-Access-Token header
        $xToken = $this->request->getHeaderLine('X-Access-Token');
        if (!empty($xToken)) {
            return $xToken;
        }

        return null;
    }

    /**
     * JWT Service instance al
     */
    private function getJWTService(): JWTService
    {
        if ($this->jwtService === null) {
            $this->jwtService = new JWTService();
        }
        return $this->jwtService;
    }
}
