<?php

namespace App\Services;

use App\Libraries\JWT;
use App\Libraries\TokenExpiredException;
use App\Libraries\TokenInvalidException;
use App\Libraries\SignatureInvalidException;
use App\Models\JwtTokenModel;

/**
 * ============================================================================
 * JWT SERVICE
 * ============================================================================
 * 
 * Token oluşturma, doğrulama, yenileme ve iptal işlemleri
 * 
 * @package    Satisahazir
 * @version    2.0.0
 * ============================================================================
 */
class JWTService
{
    private JWT $jwt;
    private JwtTokenModel $tokenModel;
    
    private string $secretKey;
    private string $issuer;
    private int $accessExpiry;
    private int $refreshExpiry;
    private string $algorithm = 'HS256';

    public function __construct()
    {
        $this->jwt = new JWT();
        $this->tokenModel = new JwtTokenModel();
        
        // Config
        $this->secretKey = getenv('JWT_SECRET_KEY') ?: $this->generateSecretKey();
        $this->issuer = getenv('JWT_ISSUER') ?: 'satisahazir.com';
        $this->accessExpiry = (int) (getenv('JWT_ACCESS_EXPIRY') ?: 900); // 15 dakika
        $this->refreshExpiry = (int) (getenv('JWT_REFRESH_EXPIRY') ?: 2592000); // 30 gün

        if (strlen($this->secretKey) < 32) {
            throw new \RuntimeException('JWT_SECRET_KEY en az 32 karakter olmalı');
        }
    }

    // =========================================================================
    // TOKEN GENERATION
    // =========================================================================

    /**
     * Access ve Refresh token çifti oluştur
     */
    public function generateTokenPair(
        array $userData,
        string $deviceUuid,
        string $platform,
        string $ipAddress,
        string $userAgent
    ): array {
        $now = time();
        $jti = $this->generateJti();

        // Access Token
        $accessPayload = [
            'iss' => $this->issuer,
            'sub' => (string) $userData['id'],
            'iat' => $now,
            'exp' => $now + $this->accessExpiry,
            'jti' => $jti,
            'type' => 'access',
            'user' => [
                'id' => (int) $userData['id'],
                'email' => $userData['email'],
                'type' => $userData['type'] ?? 'USER',
                'verified' => (bool) ($userData['verified'] ?? false),
            ],
        ];

        $accessToken = $this->jwt->encode($accessPayload, $this->secretKey, $this->algorithm);

        // Refresh Token
        $refreshJti = $this->generateJti();
        $refreshPayload = [
            'iss' => $this->issuer,
            'sub' => (string) $userData['id'],
            'iat' => $now,
            'exp' => $now + $this->refreshExpiry,
            'jti' => $refreshJti,
            'type' => 'refresh',
            'access_jti' => $jti,
        ];

        $refreshToken = $this->jwt->encode($refreshPayload, $this->secretKey, $this->algorithm);

        // Veritabanına kaydet
        $this->tokenModel->insert([
            'user_id' => $userData['id'],
            'access_token_hash' => hash('sha256', $accessToken),
            'refresh_token_hash' => hash('sha256', $refreshToken),
            'access_jti' => $jti,
            'refresh_jti' => $refreshJti,
            'device_uuid' => $deviceUuid,
            'device_platform' => $platform,
            'ip_address' => $ipAddress,
            'user_agent' => substr($userAgent, 0, 500),
            'access_expires_at' => date('Y-m-d H:i:s', $now + $this->accessExpiry),
            'refresh_expires_at' => date('Y-m-d H:i:s', $now + $this->refreshExpiry),
            'created_at' => date('Y-m-d H:i:s'),
        ]);

        return [
            'access_token' => $accessToken,
            'refresh_token' => $refreshToken,
            'expires_in' => $this->accessExpiry,
            'refresh_expires_in' => $this->refreshExpiry,
            'access_expires_at' => $now + $this->accessExpiry,
            'refresh_expires_at' => $now + $this->refreshExpiry,
            'token_type' => 'Bearer',
        ];
    }

    // =========================================================================
    // TOKEN VALIDATION
    // =========================================================================

    /**
     * Access token doğrula
     */
    public function validateAccessToken(string $token): ?array
    {
        try {
            $payload = $this->jwt->decode($token, $this->secretKey);

            // Type kontrolü
            if (($payload['type'] ?? '') !== 'access') {
                throw new TokenInvalidException('Bu bir access token değil');
            }

            // Veritabanı kontrolü (revoke edilmiş mi?)
            $hash = hash('sha256', $token);
            $tokenRecord = $this->tokenModel->findByAccessHash($hash);

            if (!$tokenRecord) {
                throw new TokenInvalidException('Token bulunamadı');
            }

            if ($tokenRecord['revoked_at'] !== null) {
                throw new TokenInvalidException('Token iptal edilmiş');
            }

            return $payload;

        } catch (TokenExpiredException $e) {
            throw $e;
        } catch (SignatureInvalidException $e) {
            throw $e;
        } catch (TokenInvalidException $e) {
            throw $e;
        } catch (\Exception $e) {
            log_message('error', 'JWT validation error: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Refresh token doğrula
     */
    public function validateRefreshToken(string $token): ?array
    {
        try {
            $payload = $this->jwt->decode($token, $this->secretKey);

            if (($payload['type'] ?? '') !== 'refresh') {
                throw new TokenInvalidException('Bu bir refresh token değil');
            }

            // Veritabanı kontrolü
            $hash = hash('sha256', $token);
            $tokenRecord = $this->tokenModel->findByRefreshHash($hash);

            if (!$tokenRecord) {
                throw new TokenInvalidException('Refresh token bulunamadı');
            }

            if ($tokenRecord['revoked_at'] !== null) {
                throw new TokenInvalidException('Refresh token iptal edilmiş');
            }

            return [
                'payload' => $payload,
                'record' => $tokenRecord,
            ];

        } catch (\Exception $e) {
            log_message('debug', 'Refresh token validation failed: ' . $e->getMessage());
            return null;
        }
    }

    // =========================================================================
    // TOKEN REFRESH
    // =========================================================================

    /**
     * Refresh token ile yeni token çifti oluştur
     */
    public function refreshTokens(string $refreshToken): ?array
    {
        $validation = $this->validateRefreshToken($refreshToken);
        
        if (!$validation) {
            return null;
        }

        $tokenRecord = $validation['record'];
        $userId = $tokenRecord['user_id'];

        // User bilgisini al
        $db = \Config\Database::connect();
        $user = $db->table('users')
            ->select('id, user_email as email, type, email_verified as verified, display_name')
            ->where('id', $userId)
            ->get()
            ->getRowArray();

        if (!$user) {
            return null;
        }

        // Eski token'ı iptal et
        $this->tokenModel->revokeToken($tokenRecord['id']);

        // Yeni token çifti oluştur
        return $this->generateTokenPair(
            [
                'id' => $user['id'],
                'email' => $user['email'],
                'type' => $user['type'],
                'verified' => (bool) $user['verified'],
            ],
            $tokenRecord['device_uuid'],
            $tokenRecord['device_platform'],
            $tokenRecord['ip_address'],
            $tokenRecord['user_agent']
        );
    }

    // =========================================================================
    // TOKEN REVOCATION
    // =========================================================================

    /**
     * Token'ı iptal et
     */
    public function revokeToken(string $token): bool
    {
        $hash = hash('sha256', $token);
        $tokenRecord = $this->tokenModel->findByAccessHash($hash);

        if ($tokenRecord) {
            return $this->tokenModel->revokeToken($tokenRecord['id']);
        }

        return false;
    }

    /**
     * Kullanıcının tüm token'larını iptal et
     */
    public function revokeAllUserTokens(int $userId): bool
    {
        return $this->tokenModel->revokeAllUserTokens($userId);
    }

    /**
     * Belirli bir cihazın token'ını iptal et
     */
    public function revokeDeviceToken(int $userId, string $deviceUuid): bool
    {
        return $this->tokenModel->revokeDeviceToken($userId, $deviceUuid);
    }

    // =========================================================================
    // SESSION MANAGEMENT
    // =========================================================================

    /**
     * Kullanıcının aktif oturumlarını getir
     */
    public function getUserSessions(int $userId): array
    {
        return $this->tokenModel->getUserActiveSessions($userId);
    }

    /**
     * Oturum sonlandır
     */
    public function terminateSession(int $userId, int $sessionId): bool
    {
        $token = $this->tokenModel->find($sessionId);
        
        if (!$token || $token['user_id'] != $userId) {
            return false;
        }

        return $this->tokenModel->revokeToken($sessionId);
    }

    // =========================================================================
    // HELPERS
    // =========================================================================

    /**
     * Unique JTI oluştur
     */
    private function generateJti(): string
    {
        return bin2hex(random_bytes(16));
    }

    /**
     * Geçici secret key oluştur (sadece development için)
     */
    private function generateSecretKey(): string
    {
        log_message('warning', 'JWT_SECRET_KEY tanımlı değil! Geçici key kullanılıyor.');
        return hash('sha256', 'temporary-insecure-key-' . APPPATH);
    }

    /**
     * Token'ın kalan süresini hesapla (saniye)
     */
    public function getTokenRemainingTime(string $token): int
    {
        $payload = $this->jwt->getPayloadWithoutVerification($token);
        if (!$payload || !isset($payload['exp'])) {
            return 0;
        }
        
        $remaining = $payload['exp'] - time();
        return max(0, $remaining);
    }
}
