<?php

namespace App\Models;

use CodeIgniter\Model;

/**
 * JWT Token Model
 * 
 * logs veritabanındaki jwt_tokens tablosu için
 */
class JwtTokenModel extends Model
{
    protected $DBGroup = 'logs'; // logs veritabanı
    protected $table = 'jwt_tokens';
    protected $primaryKey = 'id';
    protected $useAutoIncrement = true;
    protected $returnType = 'array';
    protected $useSoftDeletes = false;
    protected $useTimestamps = false;

    protected $allowedFields = [
        'user_id',
        'access_token_hash',
        'refresh_token_hash',
        'access_jti',
        'refresh_jti',
        'device_uuid',
        'device_platform',
        'device_name',
        'ip_address',
        'user_agent',
        'access_expires_at',
        'refresh_expires_at',
        'last_used_at',
        'revoked_at',
        'revoked_reason',
        'created_at',
    ];

    /**
     * Access token hash ile bul
     */
    public function findByAccessHash(string $hash): ?array
    {
        return $this->where('access_token_hash', $hash)->first();
    }

    /**
     * Refresh token hash ile bul
     */
    public function findByRefreshHash(string $hash): ?array
    {
        return $this->where('refresh_token_hash', $hash)->first();
    }

    /**
     * Token'ı iptal et
     */
    public function revokeToken(int $tokenId, string $reason = 'manual'): bool
    {
        return $this->update($tokenId, [
            'revoked_at' => date('Y-m-d H:i:s'),
            'revoked_reason' => $reason,
        ]);
    }

    /**
     * Kullanıcının tüm token'larını iptal et
     */
    public function revokeAllUserTokens(int $userId, string $reason = 'logout_all'): bool
    {
        return $this->where('user_id', $userId)
            ->where('revoked_at', null)
            ->set([
                'revoked_at' => date('Y-m-d H:i:s'),
                'revoked_reason' => $reason,
            ])
            ->update();
    }

    /**
     * Belirli cihazın token'ını iptal et
     */
    public function revokeDeviceToken(int $userId, string $deviceUuid): bool
    {
        return $this->where('user_id', $userId)
            ->where('device_uuid', $deviceUuid)
            ->where('revoked_at', null)
            ->set([
                'revoked_at' => date('Y-m-d H:i:s'),
                'revoked_reason' => 'device_logout',
            ])
            ->update();
    }

    /**
     * Kullanıcının aktif oturumlarını getir
     */
    public function getUserActiveSessions(int $userId): array
    {
        return $this->select('id, device_uuid, device_platform, device_name, ip_address, user_agent, last_used_at, created_at')
            ->where('user_id', $userId)
            ->where('revoked_at', null)
            ->where('refresh_expires_at >', date('Y-m-d H:i:s'))
            ->orderBy('last_used_at', 'DESC')
            ->findAll();
    }

    /**
     * Son kullanım zamanını güncelle
     */
    public function updateLastUsed(int $tokenId): bool
    {
        return $this->update($tokenId, [
            'last_used_at' => date('Y-m-d H:i:s'),
        ]);
    }

    /**
     * Süresi dolmuş token'ları temizle
     */
    public function cleanupExpiredTokens(): int
    {
        $builder = $this->db->table($this->table);
        
        return $builder->where('refresh_expires_at <', date('Y-m-d H:i:s'))
            ->delete();
    }

    /**
     * Token istatistikleri
     */
    public function getStatistics(): array
    {
        $now = date('Y-m-d H:i:s');
        
        return [
            'total' => $this->countAll(),
            'active' => $this->where('revoked_at', null)
                ->where('refresh_expires_at >', $now)
                ->countAllResults(),
            'revoked' => $this->where('revoked_at !=', null)->countAllResults(),
            'expired' => $this->where('refresh_expires_at <', $now)->countAllResults(),
        ];
    }
}
