<?php

namespace App\Filters;

use CodeIgniter\Filters\FilterInterface;
use CodeIgniter\HTTP\RequestInterface;
use CodeIgniter\HTTP\ResponseInterface;
use App\Services\JWTService;

/**
 * JWT Authentication Filter
 * 
 * Tüm korumalı endpoint'ler için JWT token doğrulama
 */
class JWTAuthFilter implements FilterInterface
{
    public function before(RequestInterface $request, $arguments = null)
    {
        // CORS preflight
        if ($request->getMethod() === 'options') {
            return $this->corsResponse();
        }

        // Token'ı al
        $token = $this->extractToken($request);

        if (empty($token)) {
            return $this->errorResponse('Token gerekli', 'token_required', 401);
        }

        try {
            $jwtService = new JWTService();
            $payload = $jwtService->validateAccessToken($token);

            if (!$payload) {
                return $this->errorResponse('Geçersiz token', 'token_invalid', 401);
            }

            // User bilgisini request'e ekle - Controller'lar buradan okuyacak
            $request->jwt_user = $payload['user'] ?? null;
            $request->jwt_payload = $payload;
            $request->jwt_token = $token;

            // Token kullanım zamanını güncelle
            $this->updateTokenUsage($token);

            // Token yakında dolacaksa header ile bildir
            if ($this->isTokenExpiringSoon($payload)) {
                service('response')->setHeader('X-Token-Refresh-Suggested', 'true');
            }

            return $request;

        } catch (\App\Libraries\TokenExpiredException $e) {
            return $this->errorResponse('Token süresi dolmuş', 'token_expired', 401);
        } catch (\App\Libraries\TokenInvalidException $e) {
            return $this->errorResponse('Geçersiz token', 'token_invalid', 401);
        } catch (\App\Libraries\SignatureInvalidException $e) {
            return $this->errorResponse('Token imzası geçersiz', 'signature_invalid', 401);
        } catch (\Exception $e) {
            log_message('error', 'JWT Filter Error: ' . $e->getMessage());
            return $this->errorResponse('Token doğrulama hatası', 'auth_error', 401);
        }
    }

    public function after(RequestInterface $request, ResponseInterface $response, $arguments = null)
    {
        // CORS headers ekle
        return $response
            ->setHeader('Access-Control-Allow-Origin', '*')
            ->setHeader('Access-Control-Allow-Methods', 'GET, POST, PUT, DELETE, OPTIONS')
            ->setHeader('Access-Control-Allow-Headers', 'Content-Type, Authorization, X-Requested-With, X-Access-Token')
            ->setHeader('Access-Control-Max-Age', '86400');
    }

    /**
     * Token'ı request'ten çıkar
     */
    private function extractToken(RequestInterface $request): ?string
    {
        // 1. Authorization header (öncelikli)
        $authHeader = $request->getHeaderLine('Authorization');
        if (!empty($authHeader) && preg_match('/Bearer\s+(.+)$/i', $authHeader, $matches)) {
            return $matches[1];
        }

        // 2. X-Access-Token header
        $xToken = $request->getHeaderLine('X-Access-Token');
        if (!empty($xToken)) {
            return $xToken;
        }

        // 3. Query parameter (sadece GET için, güvenlik için önerilmez)
        // $token = $request->getGet('token');
        // if (!empty($token)) return $token;

        return null;
    }

    /**
     * Token kullanım zamanını güncelle
     */
    private function updateTokenUsage(string $token): void
    {
        try {
            $tokenModel = new \App\Models\JwtTokenModel();
            $hash = hash('sha256', $token);
            $tokenRecord = $tokenModel->findByAccessHash($hash);

            if ($tokenRecord) {
                $tokenModel->updateLastUsed($tokenRecord['id']);
            }
        } catch (\Exception $e) {
            // Sessizce devam et - token update kritik değil
        }
    }

    /**
     * Token yakında dolacak mı? (5 dakika içinde)
     */
    private function isTokenExpiringSoon(array $payload): bool
    {
        $exp = $payload['exp'] ?? 0;
        $now = time();
        $threshold = 300; // 5 dakika

        return ($exp - $now) < $threshold;
    }

    /**
     * Error response
     */
    private function errorResponse(string $message, string $code, int $httpCode): ResponseInterface
    {
        return service('response')
            ->setStatusCode($httpCode)
            ->setHeader('Content-Type', 'application/json')
            ->setBody(json_encode([
                'code' => $code,
                'message' => $message,
            ]));
    }

    /**
     * CORS preflight response
     */
    private function corsResponse(): ResponseInterface
    {
        return service('response')
            ->setStatusCode(200)
            ->setHeader('Access-Control-Allow-Origin', '*')
            ->setHeader('Access-Control-Allow-Methods', 'GET, POST, PUT, DELETE, OPTIONS')
            ->setHeader('Access-Control-Allow-Headers', 'Content-Type, Authorization, X-Requested-With, X-Access-Token')
            ->setHeader('Access-Control-Max-Age', '86400');
    }
}
