<?php

namespace App\Controllers;
use CodeIgniter\RESTful\ResourceController;
use CodeIgniter\API\ResponseTrait;

class SearchRequestController extends ResourceController
{
    use ResponseTrait;
    
    /**
     * Submit a new property search request
     */
    public function propertySearchRequest()
    {
        $data = array(
            'code'	    => 'empty_code',
            'message'   => 'Empty Message'
        );

        $postBody = file_get_contents('php://input');
        $data_json = json_decode($postBody);

        $validation = \Config\Services::validation();

        $validation->setRules([
            'name' => ['label' => 'name', 'rules' => 'required'],
            'email' => ['label' => 'email', 'rules' => ['required', 'valid_email']],
            'phone' => ['label' => 'phone', 'rules' => ['required']],
            'message' => ['label' => 'message', 'rules' => ['required']],
            'search_criteria' => ['label' => 'search_criteria', 'rules' => ['required']],
            'consents' => ['label' => 'consents', 'rules' => ['required']],
        ]);

        if ($validation->withRequest($this->request)->run()) {
            $data_json = $this->request->getJSON();

            $searchRequestModel = model('App\Models\SearchRequestModel');
            
            // Parse consents
            $consents = json_decode($data_json->consents, true);
            
            // Check required consents
            if (!isset($consents['data_sharing_consent']) || !$consents['data_sharing_consent']) {
                return $this->respond(array(
                    'code'	    => 'failed',
                    'message'   => 'Veri paylaşım izni gereklidir'
                ));
            }
            
            if (!isset($consents['contact_permission_consent']) || !$consents['contact_permission_consent']) {
                return $this->respond(array(
                    'code'	    => 'failed',
                    'message'   => 'İletişim izni gereklidir'
                ));
            }
            
            // Save search request to database
            $requestData = array();
            $requestData['name'] = $data_json->name;
            $requestData['email'] = $data_json->email;
            $requestData['phone'] = $data_json->phone;
            $requestData['message'] = $data_json->message;
            $requestData['search_criteria'] = $data_json->search_criteria;
            $requestData['consents'] = $data_json->consents;
            $requestData['consent_timestamp'] = $data_json->consent_timestamp ?? date('Y-m-d H:i:s');
            $requestData['status'] = 'pending';
            $requestData['readed'] = 0;
            $requestData['date'] = date('Y-m-d H:i:s');

            $insert_id = $searchRequestModel->insert($requestData);
            
            // Send notifications for new search request
            if ($insert_id) {
                $requestData['id'] = $insert_id;
                $searchRequestModel->sendNewSearchRequestNotifications($requestData);
            }

            return $this->respond(array(
                'code'	    => 'success',
                'message'   => 'Mülk arama talebiniz başarıyla alındı. En kısa sürede sizinle iletişime geçeceğiz.'
            ));
        }
        else
        {
            return $this->respond(array(
                'code'	    => 'failed',
                'message'   => join("\n", $validation->getErrors())
            ));
        }       

        return $this->respond($data);
    }
    
    /**
     * Get search requests list based on user role
     * ✅ GÜNCELLENMİŞ: Lokasyon ve kategori isimlerini backend'de çözüyor
     */
    public function getSearchRequests()
    {
        // Check authentication
        $tokenModel = model('App\Models\TokenModel');
        $userModel = model('App\Models\UserModel');
        
        // [Check Token]
        if (empty($_GET['token'])) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Token eksik'
            ]);
        }

        $token = $_GET['token'];
        $user_token = $tokenModel->where('token', $token)->findAll(1);

        if (count($user_token) == 0) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Geçerli token eksik'
            ]);
        }
        
        $userId = $user_token[0]['user_id'];
        
        // Get user info to check role
        $user = $userModel->find($userId);
        
        if (!$user) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Kullanıcı bulunamadı'
            ]);
        }
        
        // Check if user is admin or agent
        $isAdmin = $user['type'] === 'ADMIN';
        $isAgent = $user['type'] === 'AGENT';
        
        if (!$isAdmin && !$isAgent) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Bu işlem için yetkiniz yok'
            ]);
        }
        
        // Pagination params
        $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 20;
        $offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
        
        // Get search requests
        $searchRequestModel = model('App\Models\SearchRequestModel');
        $requests = $searchRequestModel->getSearchRequestsWithDetails($userId, $isAdmin, $limit, $offset);
        
        // ✅ YENİ: Her request için lokasyon ve kategori isimlerini çöz
        $db = \Config\Database::connect();
        
        foreach ($requests as &$request) {
            // Search criteria'yı parse et
            $searchCriteria = json_decode($request['search_criteria'], true);
            
            if (!empty($searchCriteria)) {
                // ✅ Lokasyon ID'lerini isimlere çevir
                $request['location_names'] = $this->resolveLocationNames($searchCriteria, $db);
                
                // ✅ Kategori ID'sini isme çevir
                $request['category_name'] = $this->resolveCategoryName($searchCriteria, $db);
                
                // ✅ Diğer önemli kriterler
                $request['price_range'] = $this->formatPriceRange($searchCriteria);
                $request['room_count'] = $searchCriteria['field_7'] ?? '';
                $request['bathroom_count'] = $searchCriteria['field_8'] ?? '';
                $request['size'] = $searchCriteria['field_9'] ?? '';
                $request['search_term'] = $searchCriteria['search'] ?? '';
            } else {
                $request['location_names'] = '';
                $request['category_name'] = '';
                $request['price_range'] = '';
                $request['room_count'] = '';
                $request['bathroom_count'] = '';
                $request['size'] = '';
                $request['search_term'] = '';
            }
        }
        
        // Get unread count
        $unreadCount = $searchRequestModel->getUnreadCount($userId, $isAdmin);
        
        // Separate requests into categories
        $categorizedRequests = [
            'pending' => [],
            'assigned' => [],
            'completed' => []
        ];
        
        foreach ($requests as $request) {
            if ($request['status'] === 'completed') {
                $categorizedRequests['completed'][] = $request;
            } elseif ($request['status'] === 'assigned') {
                $categorizedRequests['assigned'][] = $request;
            } else {
                $categorizedRequests['pending'][] = $request;
            }
        }
        
        if(getenv('CI_ENVIRONMENT') == 'development') {
            log_message('info', '✅ Search Requests: ' . count($requests) . ' talep yüklendi');
            log_message('info', '   Lokasyon çözümleme aktif');
        }
        
        return $this->respond([
            'code'     => 'success',
            'message'  => 'Arama istekleri başarıyla alındı',
            'requests' => $requests,
            'categorized_requests' => $categorizedRequests,
            'unread_count' => $unreadCount,
            'total' => count($requests),
        ]);
    }
    
    /**
     * Get search request details with replies
     * ✅ GÜNCELLENMİŞ: Lokasyon ve kategori isimlerini backend'de çözüyor
     */
    public function getRequestDetails()
    {
        // Check authentication
        $tokenModel = model('App\Models\TokenModel');
        $userModel = model('App\Models\UserModel');
        
        // [Check Token]
        if (empty($_GET['token'])) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Token eksik'
            ]);
        }

        $token = $_GET['token'];
        $user_token = $tokenModel->where('token', $token)->findAll(1);

        if (count($user_token) == 0) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Geçerli token eksik'
            ]);
        }
        
        $userId = $user_token[0]['user_id'];
        
        // Check request ID
        if (empty($_GET['request_id'])) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'İstek ID eksik'
            ]);
        }
        
        $requestId = (int)$_GET['request_id'];
        
        // Get user info to check role
        $user = $userModel->find($userId);
        
        if (!$user) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Kullanıcı bulunamadı'
            ]);
        }
        
        // Check if user is admin or agent
        $isAdmin = $user['type'] === 'ADMIN';
        $isAgent = $user['type'] === 'AGENT';
        
        if (!$isAdmin && !$isAgent) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Bu işlem için yetkiniz yok'
            ]);
        }
        
        // Get request details
        $searchRequestModel = model('App\Models\SearchRequestModel');
        $request = $searchRequestModel->getRequestWithReplies($requestId, $userId, $isAdmin);
        
        if (!$request) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'İstek bulunamadı veya görüntüleme yetkiniz yok'
            ]);
        }
        
        // ✅ YENİ: Lokasyon ve kategori isimlerini çöz
        $db = \Config\Database::connect();
        $searchCriteria = json_decode($request['search_criteria'], true);
        
        if (!empty($searchCriteria)) {
            $request['location_names'] = $this->resolveLocationNames($searchCriteria, $db);
            $request['category_name'] = $this->resolveCategoryName($searchCriteria, $db);
            $request['price_range'] = $this->formatPriceRange($searchCriteria);
            $request['room_count'] = $searchCriteria['field_7'] ?? '';
            $request['bathroom_count'] = $searchCriteria['field_8'] ?? '';
            $request['size'] = $searchCriteria['field_9'] ?? '';
            $request['search_term'] = $searchCriteria['search'] ?? '';
        } else {
            $request['location_names'] = '';
            $request['category_name'] = '';
            $request['price_range'] = '';
            $request['room_count'] = '';
            $request['bathroom_count'] = '';
            $request['size'] = '';
            $request['search_term'] = '';
        }
        
        if(getenv('CI_ENVIRONMENT') == 'development') {
            log_message('info', '✅ Request Details: ID ' . $requestId);
            log_message('info', '   Lokasyonlar: ' . $request['location_names']);
            log_message('info', '   Kategori: ' . $request['category_name']);
        }
        
        return $this->respond([
            'code'     => 'success',
            'message'  => 'İstek detayları başarıyla alındı',
            'request' => $request,
            'is_admin' => $isAdmin
        ]);
    }
    
    /**
     * ✅ YENİ: Lokasyon ID'lerinden tam adres oluştur
     * Çoklu lokasyon desteği (virgülle ayrılmış)
     */
    private function resolveLocationNames($criteria, $db)
    {
        $locationId = $criteria['search_location'] ?? $criteria['location_id'] ?? '';
        
        if (empty($locationId)) {
            return '';
        }
        
        // Virgülle ayrılmış ID'ler
        $locationIds = explode(',', $locationId);
        $locationNames = [];
        
        foreach ($locationIds as $id) {
            $id = trim($id);
            if (empty($id)) continue;
            
            $fullPath = $this->getLocationFullPath($id, $db);
            if ($fullPath) {
                $locationNames[] = $fullPath;
            }
        }
        
        if (empty($locationNames)) {
            return '';
        }
        
        // Birden fazla lokasyon varsa kısalt
        if (count($locationNames) == 1) {
            return $locationNames[0];
        } elseif (count($locationNames) == 2) {
            return $locationNames[0] . ', ' . $locationNames[1];
        } else {
            return $locationNames[0] . ', ' . $locationNames[1] . ' +' . (count($locationNames) - 2);
        }
    }
    
    /**
     * ✅ YENİ: Lokasyon ID'den tam yol (İl / İlçe / Mahalle)
     * ✅ DÜZELTİLDİ: Direkt DB sorgusu kullanıyor
     */
    private function getLocationFullPath($locationId, $db)
    {
        try {
            // treefield ve treefield_lang tablolarını join et
            $location = $db->table('treefield t')
                ->select('t.id, t.level, t.parent_id, tl.value')
                ->join('treefield_lang tl', 't.id = tl.treefield_id', 'left')
                ->where('t.id', $locationId)
                ->where('t.field_id', 64) // Location field ID
                ->where('tl.language_id', getenv('LANGUAGE_ID'))
                ->get()
                ->getRowArray();
            
            if (!$location) {
                if(getenv('CI_ENVIRONMENT') == 'development') {
                    log_message('warning', "⚠️ Lokasyon bulunamadı: ID $locationId");
                }
                return $locationId;
            }
            
            $level = intval($location['level']);
            $path = [$location['value']];
            
            // Mahalle (Level 2) → İlçe + İl ekle
            if ($level === 2 && !empty($location['parent_id'])) {
                $district = $db->table('treefield t')
                    ->select('t.parent_id, tl.value')
                    ->join('treefield_lang tl', 't.id = tl.treefield_id', 'left')
                    ->where('t.id', $location['parent_id'])
                    ->where('t.field_id', 64)
                    ->where('tl.language_id', getenv('LANGUAGE_ID'))
                    ->get()
                    ->getRowArray();
                
                if ($district) {
                    $path[] = $district['value'];
                    
                    // İl ekle
                    if (!empty($district['parent_id'])) {
                        $province = $db->table('treefield t')
                            ->select('tl.value')
                            ->join('treefield_lang tl', 't.id = tl.treefield_id', 'left')
                            ->where('t.id', $district['parent_id'])
                            ->where('t.field_id', 64)
                            ->where('tl.language_id', getenv('LANGUAGE_ID'))
                            ->get()
                            ->getRowArray();
                        
                        if ($province) {
                            $path[] = $province['value'];
                        }
                    }
                }
            }
            // İlçe (Level 1) → İl ekle
            elseif ($level === 1 && !empty($location['parent_id'])) {
                $province = $db->table('treefield t')
                    ->select('tl.value')
                    ->join('treefield_lang tl', 't.id = tl.treefield_id', 'left')
                    ->where('t.id', $location['parent_id'])
                    ->where('t.field_id', 64)
                    ->where('tl.language_id', getenv('LANGUAGE_ID'))
                    ->get()
                    ->getRowArray();
                
                if ($province) {
                    $path[] = $province['value'];
                }
            }
            
            // Ters çevir: İl / İlçe / Mahalle formatı
            $fullPath = implode(' / ', array_reverse($path));
            
            if(getenv('CI_ENVIRONMENT') == 'development') {
                log_message('info', "📍 Lokasyon çözümlendi: $locationId → $fullPath");
            }
            
            return $fullPath;
            
        } catch (\Exception $e) {
            log_message('error', 'Location path error: ' . $e->getMessage());
            return $locationId;
        }
    }
    
    /**
     * ✅ YENİ: Kategori ID'den isim
     * ✅ DÜZELTİLDİ: Direkt DB sorgusu kullanıyor
     */
    private function resolveCategoryName($criteria, $db)
    {
        $categoryId = $criteria['search_category'] ?? $criteria['category_id'] ?? '';
        
        if (empty($categoryId)) {
            return '';
        }
        
        try {
            $category = $db->table('treefield t')
                ->select('tl.value')
                ->join('treefield_lang tl', 't.id = tl.treefield_id', 'left')
                ->where('t.id', $categoryId)
                ->where('t.field_id', 79) // Category field ID
                ->where('tl.language_id', getenv('LANGUAGE_ID'))
                ->get()
                ->getRowArray();
            
            $categoryName = $category ? $category['value'] : $categoryId;
            
            if(getenv('CI_ENVIRONMENT') == 'development') {
                log_message('info', "📦 Kategori çözümlendi: $categoryId → $categoryName");
            }
            
            return $categoryName;
        } catch (\Exception $e) {
            log_message('error', 'Category resolve error: ' . $e->getMessage());
            return $categoryId;
        }
    }
    
    /**
     * ✅ YENİ: Fiyat aralığı formatla
     */
    private function formatPriceRange($criteria)
    {
        $min = $criteria['field_6_min'] ?? '';
        $max = $criteria['field_6_max'] ?? '';
        
        if (empty($min) && empty($max)) {
            return '';
        }
        
        $parts = [];
        if (!empty($min)) {
            $minFormatted = '€' . number_format(intval($min), 0, ',', '.');
            $parts[] = $minFormatted;
        }
        if (!empty($max)) {
            $maxFormatted = '€' . number_format(intval($max), 0, ',', '.');
            $parts[] = $maxFormatted;
        }
        
        return implode(' - ', $parts);
    }
    
    /**
     * Assign request to agent (Admin only)
     */
    public function assignToAgent()
    {
        // Check authentication
        $tokenModel = model('App\Models\TokenModel');
        $userModel = model('App\Models\UserModel');
        
        $data_json = $this->request->getJSON();
        
        // [Check Token]
        if (empty($data_json->token)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Token eksik'
            ]);
        }

        $token = $data_json->token;
        $user_token = $tokenModel->where('token', $token)->findAll(1);

        if (count($user_token) == 0) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Geçerli token eksik'
            ]);
        }
        
        $userId = $user_token[0]['user_id'];
        
        // Check required parameters
        if (empty($data_json->request_id) || empty($data_json->agent_id)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'İstek ID ve Temsilci ID zorunludur'
            ]);
        }
        
        $requestId = (int)$data_json->request_id;
        $agentId = (int)$data_json->agent_id;
        $note = $data_json->note ?? '';
        
        // Get user info to check role
        $user = $userModel->find($userId);
        
        if (!$user) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Kullanıcı bulunamadı'
            ]);
        }
        
        // Check if user is admin
        if ($user['type'] !== 'ADMIN') {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Bu işlem sadece yöneticiler tarafından yapılabilir'
            ]);
        }
        
        // Assign the request
        $searchRequestModel = model('App\Models\SearchRequestModel');
        $success = $searchRequestModel->assignToAgent($requestId, $agentId, $note, $userId);
        
        if (!$success) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'İstek atanırken bir hata oluştu'
            ]);
        }
        
        return $this->respond([
            'code'     => 'success',
            'message'  => 'İstek başarıyla temsilciye atandı'
        ]);
    }

    /**
     * Add a reply to a search request
     */
    public function addReply()
    {
        // Check authentication
        $tokenModel = model('App\Models\TokenModel');
        $userModel = model('App\Models\UserModel');
        
        $data_json = $this->request->getJSON();
        
        // [Check Token]
        if (empty($data_json->token)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Token eksik'
            ]);
        }

        $token = $data_json->token;
        $user_token = $tokenModel->where('token', $token)->findAll(1);

        if (count($user_token) == 0) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Geçerli token eksik'
            ]);
        }
        
        $userId = $user_token[0]['user_id'];
        
        // Check required parameters
        if (empty($data_json->request_id) || empty($data_json->message)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'İstek ID ve mesaj alanları zorunludur'
            ]);
        }
        
        $requestId = (int)$data_json->request_id;
        $message = $data_json->message;
        $isSystemMessage = isset($data_json->system_message) ? (bool)$data_json->system_message : false;
        
        // Get user info to check role
        $user = $userModel->find($userId);
        
        if (!$user) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Kullanıcı bulunamadı'
            ]);
        }
        
        // Check if user is admin or agent
        $isAdmin = $user['type'] === 'ADMIN';
        $isAgent = $user['type'] === 'AGENT';
        
        if (!$isAdmin && !$isAgent) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Bu işlem için yetkiniz yok'
            ]);
        }
        
        // Add the reply
        $searchRequestModel = model('App\Models\SearchRequestModel');
        $replyId = $searchRequestModel->addReply($requestId, $userId, $message, $isSystemMessage);
        
        if (!$replyId) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Cevap eklenirken bir hata oluştu veya yetkiniz yok'
            ]);
        }
        
        // Get updated request details
        $request = $searchRequestModel->getRequestWithReplies($requestId, $userId, $isAdmin);
        
        return $this->respond([
            'code'     => 'success',
            'message'  => 'Cevap başarıyla eklendi',
            'reply_id' => $replyId,
            'request' => $request
        ]);
    }

    /**
     * Complete a search request
     */
    public function completeRequest()
    {
        // Check authentication
        $tokenModel = model('App\Models\TokenModel');
        $userModel = model('App\Models\UserModel');
        
        $data_json = $this->request->getJSON();
        
        // [Check Token]
        if (empty($data_json->token)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Token eksik'
            ]);
        }

        $token = $data_json->token;
        $user_token = $tokenModel->where('token', $token)->findAll(1);

        if (count($user_token) == 0) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Geçerli token eksik'
            ]);
        }
        
        $userId = $user_token[0]['user_id'];
        
        // Check required parameters
        if (empty($data_json->request_id)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'İstek ID zorunludur'
            ]);
        }
        
        $requestId = (int)$data_json->request_id;
        
        // Get user info to check role
        $user = $userModel->find($userId);
        
        if (!$user) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Kullanıcı bulunamadı'
            ]);
        }
        
        // Check if user is admin or agent
        $isAdmin = $user['type'] === 'ADMIN';
        $isAgent = $user['type'] === 'AGENT';
        
        if (!$isAdmin && !$isAgent) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Bu işlem için yetkiniz yok'
            ]);
        }
        
        // Complete the request
        $searchRequestModel = model('App\Models\SearchRequestModel');
        $success = $searchRequestModel->completeRequest($requestId, $userId);
        
        if (!$success) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'İstek tamamlanırken bir hata oluştu veya yetkiniz yok'
            ]);
        }
        
        return $this->respond([
            'code'     => 'success',
            'message'  => 'İstek başarıyla tamamlandı'
        ]);
    }

    /**
     * Get assigned requests for agent
     */
    public function getMyRequests()
    {
        // Check authentication
        $tokenModel = model('App\Models\TokenModel');
        $userModel = model('App\Models\UserModel');
        
        // [Check Token]
        if (empty($_GET['token'])) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Token eksik'
            ]);
        }

        $token = $_GET['token'];
        $user_token = $tokenModel->where('token', $token)->findAll(1);

        if (count($user_token) == 0) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Geçerli token eksik'
            ]);
        }
        
        $userId = $user_token[0]['user_id'];
        
        // Get user info to check role
        $user = $userModel->find($userId);
        
        if (!$user) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Kullanıcı bulunamadı'
            ]);
        }
        
        // Check if user is agent
        if ($user['type'] !== 'AGENT') {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Bu işlem sadece temsilciler için geçerlidir'
            ]);
        }
        
        // Pagination params
        $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 20;
        $offset = isset($_GET['offset']) ? (int)$_GET['offset'] : 0;
        
        // Get assigned requests
        $searchRequestModel = model('App\Models\SearchRequestModel');
        $requests = $searchRequestModel->getAssignedRequests($userId, $limit, $offset);
        
        // Get unread count
        $unreadCount = $searchRequestModel->getUnreadCount($userId, false);
        
        return $this->respond([
            'code'     => 'success',
            'message'  => 'Atanmış istekler başarıyla alındı',
            'requests' => $requests,
            'unread_count' => $unreadCount,
            'total' => count($requests),
        ]);
    }
    
    /**
     * Update search criteria
     */
    public function updateCriteria()
    {
        // Check authentication
        $tokenModel = model('App\Models\TokenModel');
        $userModel = model('App\Models\UserModel');
        
        $data_json = $this->request->getJSON();
        
        // [Check Token]
        if (empty($data_json->token)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Token eksik'
            ]);
        }

        $token = $data_json->token;
        $user_token = $tokenModel->where('token', $token)->findAll(1);

        if (count($user_token) == 0) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Geçerli token eksik'
            ]);
        }
        
        $userId = $user_token[0]['user_id'];
        
        // Check required parameters
        if (empty($data_json->request_id) || empty($data_json->search_criteria)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'İstek ID ve arama kriterleri zorunludur'
            ]);
        }
        
        $requestId = (int)$data_json->request_id;
        $searchCriteria = $data_json->search_criteria;
        
        // Get user info to check role
        $user = $userModel->find($userId);
        
        if (!$user) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Kullanıcı bulunamadı'
            ]);
        }
        
        // Check if user is admin or agent
        $isAdmin = $user['type'] === 'ADMIN';
        $isAgent = $user['type'] === 'AGENT';
        
        if (!$isAdmin && !$isAgent) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Bu işlem için yetkiniz yok'
            ]);
        }
        
        // Update criteria
        $searchRequestModel = model('App\Models\SearchRequestModel');
        $success = $searchRequestModel->updateSearchCriteria($requestId, $searchCriteria, $userId);
        
        if (!$success) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Arama kriterleri güncellenirken bir hata oluştu veya yetkiniz yok'
            ]);
        }
        
        return $this->respond([
            'code'     => 'success',
            'message'  => 'Arama kriterleri başarıyla güncellendi'
        ]);
    }
    
    /**
     * Get unread search request count
     */
    public function getUnreadCount()
    {
        // Check authentication
        $tokenModel = model('App\Models\TokenModel');
        $userModel = model('App\Models\UserModel');
        
        // [Check Token]
        if (empty($_GET['token'])) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Token eksik'
            ]);
        }

        $token = $_GET['token'];
        $user_token = $tokenModel->where('token', $token)->findAll(1);

        if (count($user_token) == 0) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Geçerli token eksik'
            ]);
        }
        
        $userId = $user_token[0]['user_id'];
        
        // Get user info to check role
        $user = $userModel->find($userId);
        
        if (!$user) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Kullanıcı bulunamadı'
            ]);
        }
        
        // Check if user is admin or agent
        $isAdmin = $user['type'] === 'ADMIN';
        $isAgent = $user['type'] === 'AGENT';
        
        if (!$isAdmin && !$isAgent) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Bu işlem için yetkiniz yok'
            ]);
        }
        
        // Get unread count
        $searchRequestModel = model('App\Models\SearchRequestModel');
        $unreadCount = $searchRequestModel->getUnreadCount($userId, $isAdmin);
        
        return $this->respond([
            'code'     => 'success',
            'message'  => 'Okunmamış istek sayısı başarıyla alındı',
            'unread_count' => $unreadCount
        ]);
    }
}