<?php
// ForgetMeController.php

namespace App\Controllers;

use CodeIgniter\RESTful\ResourceController;
use CodeIgniter\API\ResponseTrait;

class ForgetMeController extends ResourceController
{
    use ResponseTrait;
    
    public function forgetMe()
    {
        $data = [
            'code'     => 'empty_code',
            'message'  => 'Empty Message'
        ];

        $data_json = $this->request->getJSON();
        
        // Zorunlu parametreleri kontrol et
        if (empty($data_json->user_id) || empty($data_json->token) || empty($data_json->reason)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Eksik parametreler: user_id, token ve reason alanları zorunludur'
            ]);
        }

        // Token kontrolü
        $tokenModel = model('App\Models\TokenModel');
        $token = $data_json->token;
        $user_token = $tokenModel->where('token', $token)->findAll(1);

        if (count($user_token) == 0 || $user_token[0]['user_id'] != $data_json->user_id) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Geçersiz token veya user ID'
            ]);
        }

        // Talebi kaydet
        $forgetMeModel = model('App\Models\ForgetMeModel');
        
        $forgetMeData = [
            'user_id' => $data_json->user_id,
            'reason' => $data_json->reason,
            'device_uuid' => $data_json->device_uuid ?? '',
            'status' => 'pending', // pending, approved, rejected
            'created_at' => date('Y-m-d H:i:s'),
            'ip_address' => $this->request->getIPAddress(),
            'user_agent' => $this->request->getUserAgent()->getAgentString(),
        ];
        
        try {
            $requestId = $forgetMeModel->insert($forgetMeData);
            
            if ($requestId) {
                // Yöneticilere bildirim gönder
                $notificationModel = model('App\Models\NotificationModel');
                $userModel = model('App\Models\UserModel');
                
                // Kullanıcı bilgilerini al
                $user = $userModel->find($data_json->user_id);
                $userName = $user ? $user['name_surname'] : 'Bilinmeyen kullanıcı';
                $userEmail = $user ? $user['mail'] : 'Bilinmeyen email';
                
                // Admin kullanıcıları bul
                $admins = $userModel->where('type', 'ADMIN')->findAll();
                
              /*  foreach ($admins as $admin) {
                    // Her admin için bildirim oluştur
                    $notificationData = [
                        'user_id' => $admin['id'],
                        'title' => 'Yeni GDPR Unutulma Hakkı Talebi',
                        'body' => "Kullanıcı: $userName ($userEmail) tarafından unutulma hakkı talebi oluşturuldu.",
                        'data' => json_encode([
                            'request_id' => $requestId,
                            'user_id' => $data_json->user_id,
                            'redirect_type' => 'gdpr_request',
                            'redirect_id' => $requestId
                        ]),
                        'target' => 'admin',
                        'status' => 'sent',
                        'created_by' => 0 // Sistem tarafından oluşturuldu
                    ];
                    
                    $notificationModel->addNotification($notificationData);
                }
                
                // Kullanıcıya bilgi e-postası gönder
                $this->_sendForgetMeRequestEmail($user, $requestId);
                */
                return $this->respond([
                    'code'     => 'success',
                    'message'  => 'Unutulma hakkı talebiniz başarıyla alındı. En kısa sürede incelenecektir.',
                    'request_id' => $requestId
                ]);
            } else {
                return $this->respond([
                    'code'     => 'failed',
                    'message'  => 'Talep kaydedilirken bir hata oluştu'
                ]);
            }
        } catch (\Exception $e) {
            log_message('error', 'Unutulma hakkı talebi kaydedilirken hata: ' . $e->getMessage());
            
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Talep kaydedilirken bir hata oluştu: ' . $e->getMessage()
            ]);
        }
    }
    
    // Admin için unutulma hakkı taleplerini getir
    public function getForgetMeRequests()
    {
        // Token kontrolü
        if (empty($_GET['token'])) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Token eksik'
            ]);
        }

        $token = $_GET['token'];
        $tokenModel = model('App\Models\TokenModel');
        
        $user_token = $tokenModel->where('token', $token)->findAll(1);

        if (count($user_token) == 0) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Geçersiz token'
            ]);
        }
        
        $userId = $user_token[0]['user_id'];
        
        // Admin kontrolü
        $userModel = model('App\Models\UserModel');
        $user = $userModel->find($userId);
        
        if (!$user || $user['type'] !== 'ADMIN') {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Bu işlemi gerçekleştirmek için yönetici yetkisi gereklidir'
            ]);
        }
        
        // Talepleri getir
        $forgetMeModel = model('App\Models\ForgetMeModel');
        $requests = $forgetMeModel->getRequestsWithUserInfo();
        
        return $this->respond([
            'code'     => 'success',
            'message'  => 'Unutulma hakkı talepleri başarıyla getirildi',
            'requests' => $requests
        ]);
    }
    
    // Admin için unutulma hakkı talebini işle (onay/ret)
    public function processForgetMeRequest()
    {
        $data_json = $this->request->getJSON();
        
        // Zorunlu parametreleri kontrol et
        if (empty($data_json->request_id) || empty($data_json->token) || !isset($data_json->approve)) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Eksik parametreler: request_id, token ve approve alanları zorunludur'
            ]);
        }
        
        // Token kontrolü
        $tokenModel = model('App\Models\TokenModel');
        $token = $data_json->token;
        $user_token = $tokenModel->where('token', $token)->findAll(1);

        if (count($user_token) == 0) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Geçersiz token'
            ]);
        }
        
        $adminId = $user_token[0]['user_id'];
        
        // Admin kontrolü
        $userModel = model('App\Models\UserModel');
        $admin = $userModel->find($adminId);
        
        if (!$admin || $admin['type'] !== 'ADMIN') {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Bu işlemi gerçekleştirmek için yönetici yetkisi gereklidir'
            ]);
        }
        
        // Talebi al
        $forgetMeModel = model('App\Models\ForgetMeModel');
        $request = $forgetMeModel->find($data_json->request_id);
        
        if (!$request) {
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Talep bulunamadı'
            ]);
        }
        
        try {
            // Talebi güncelle
            $updateData = [
                'status' => $data_json->approve ? 'approved' : 'rejected',
                'processed_at' => date('Y-m-d H:i:s'),
                'processed_by' => $adminId
            ];
            
            if (!$data_json->approve && !empty($data_json->reject_reason)) {
                $updateData['reject_reason'] = $data_json->reject_reason;
            }
            
            $forgetMeModel->update($data_json->request_id, $updateData);
            
            // Kullanıcıyı bilgilendir
            $user = $userModel->find($request['user_id']);
            
            if ($data_json->approve) {
                // Talep onaylandı, kullanıcı verilerini sil
                $this->_deleteUserData($request['user_id']);
                
                // Onay e-postası gönder
                $this->_sendForgetMeApprovalEmail($user);
            } else {
                // Ret e-postası gönder
                $this->_sendForgetMeRejectionEmail($user, $data_json->reject_reason ?? '');
            }
            
            return $this->respond([
                'code'     => 'success',
                'message'  => 'Unutulma hakkı talebi başarıyla güncellendi',
                'status'   => $data_json->approve ? 'approved' : 'rejected'
            ]);
        } catch (\Exception $e) {
            log_message('error', 'Unutulma hakkı talebi işlenirken hata: ' . $e->getMessage());
            
            return $this->respond([
                'code'     => 'failed',
                'message'  => 'Talep işlenirken bir hata oluştu: ' . $e->getMessage()
            ]);
        }
    }
    
    // Kullanıcıya unutulma hakkı talebi bilgi e-postası gönder
    private function _sendForgetMeRequestEmail($user, $requestId)
    {
        if (!$user || empty($user['mail'])) {
            return false;
        }
        
        $emailService = service('EmailService');
        
        $subject = 'Unutulma Hakkı Talebiniz Alındı';
        $message = "Sayın {$user['name_surname']},\n\n";
        $message .= "Unutulma hakkı talebiniz başarıyla alınmıştır. Talebiniz #$requestId referans numarası ile kaydedilmiştir.\n\n";
        $message .= "Talebiniz en kısa sürede incelenecek ve sizinle iletişime geçilecektir. Bu süreç 30 güne kadar sürebilir.\n\n";
        $message .= "Bilgilerinize,\n";
        $message .= "SatışaHazır.com Ekibi";
        
        return $emailService->sendEmail($user['mail'], $subject, $message);
    }
    
    // Kullanıcıya unutulma hakkı talebi onaylandı e-postası gönder
    private function _sendForgetMeApprovalEmail($user)
    {
        if (!$user || empty($user['mail'])) {
            return false;
        }
        
        $emailService = service('EmailService');
        
        $subject = 'Unutulma Hakkı Talebiniz Onaylandı';
        $message = "Sayın {$user['name_surname']},\n\n";
        $message .= "Unutulma hakkı talebiniz onaylanmıştır. Hesabınız ve kişisel verileriniz sistemimizden silinmiştir.\n\n";
        $message .= "Yasal yükümlülükler nedeniyle bazı sınırlı veriler belirtilen süre boyunca saklanabilir. Bu veriler, kimliğinizi belirlemek için kullanılamayacak şekilde anonim hale getirilecektir.\n\n";
        $message .= "Bizi tercih ettiğiniz için teşekkür ederiz.\n\n";
        $message .= "Saygılarımızla,\n";
        $message .= "SatışaHazır.com Ekibi";
        
        return $emailService->sendEmail($user['mail'], $subject, $message);
    }
    
    // Kullanıcıya unutulma hakkı talebi reddedildi e-postası gönder
    private function _sendForgetMeRejectionEmail($user, $rejectReason)
    {
        if (!$user || empty($user['mail'])) {
            return false;
        }
        
        $emailService = service('EmailService');
        
        $subject = 'Unutulma Hakkı Talebiniz Hakkında Bilgilendirme';
        $message = "Sayın {$user['name_surname']},\n\n";
        $message .= "Unutulma hakkı talebiniz değerlendirilmiş olup, aşağıdaki sebeplerden dolayı işleme alınamamıştır:\n\n";
        $message .= "$rejectReason\n\n";
        $message .= "Herhangi bir sorunuz varsa veya itiraz etmek isterseniz, lütfen bizimle info@satisahazir.com adresinden iletişime geçiniz.\n\n";
        $message .= "Saygılarımızla,\n";
        $message .= "SatışaHazır.com Ekibi";
        
        return $emailService->sendEmail($user['mail'], $subject, $message);
    }
    
    // Kullanıcı verilerini sil (GDPR Madde 17)
    private function _deleteUserData($userId)
    {
        $db = \Config\Database::connect();
        
        try {
            // İşlem başlat
            $db->transStart();
            
            // 1. Kullanıcı verilerini anonim hale getir
            $userModel = model('App\Models\UserModel');
            $user = $userModel->find($userId);
            
            if (!$user) {
                throw new \Exception('Kullanıcı bulunamadı');
            }
            
            // Kullanıcı temel bilgilerini anonim hale getir
            $anonymizedData = [
                'mail' => 'deleted_' . time() . '_' . substr(md5($user['mail']), 0, 8) . '@deleted.example.com',
                'username' => 'deleted_user_' . time() . '_' . $userId,
                'password' => password_hash(random_bytes(16), PASSWORD_DEFAULT), // Rastgele şifre
                'name_surname' => 'Silinmiş Kullanıcı',
                'phone' => null,
                'address' => null,
                'zip_city' => null,
                'facebook_link' => null,
                'twitter_link' => null,
                'gplus_link' => null,
                'linkedin_link' => null,
                'image_user_filename' => null,
                'auth_gmail' => null,
                'auth_uid' => null,
                'fcmToken' => null,
                'is_deleted' => 1,
                'deleted_at' => date('Y-m-d H:i:s'),
                'deletion_reason' => 'GDPR Madde 17 - Unutulma Hakkı',
            ];
            
            $userModel->update($userId, $anonymizedData);
            
            // 2. FCM Token ve cihaz bilgilerini temizle
            $deviceModel = model('App\Models\DeviceModel');
            $devices = $deviceModel->where('user_id', $userId)->findAll();
            
            foreach ($devices as $device) {
                $deviceModel->update($device['id'], [
                    'user_id' => null,
                    'email' => null,
                    'fcmToken' => null,
                    'push_token' => null,
                    'push_consent' => 0,
                    'marketing_consent' => 0,
                    'is_active' => 0,
                    'topic_subscriptions' => '[]'
                ]);
                
                // Konu aboneliklerini temizle
                $db->table('device_topic_subscriptions')
                   ->where('device_uuid', $device['device_uuid'])
                   ->delete();
            }
            
            // 3. İzin kayıtlarını temizle
            $consentModel = model('App\Models\ConsentModel');
            $consents = $consentModel->where('user_id', $userId)->findAll();
            
            foreach ($consents as $consent) {
                $consentModel->update($consent['id'], [
                    'kvkk_consent' => '0',
                    'gdpr_consent' => '0',
                    'privacy_consent' => '0',
                    'marketing_consent' => '0',
                    'analytics_consent' => '0',
                    'push_consent' => '0',
                    'device_info_consent' => '0'
                ]);
            }
            
            // 4. Bildirim geçmişini temizle
            $notificationModel = model('App\Models\NotificationModel');
            $notifications = $notificationModel->where('user_id', $userId)->findAll();
            
            foreach ($notifications as $notification) {
                $notificationModel->update($notification['id'], [
                    'body' => 'Silinmiş bildirim',
                    'data' => '{"deleted":true}',
                    'status' => 'deleted'
                ]);
            }
            
            // 5. Token bilgilerini temizle
            $tokenModel = model('App\Models\TokenModel');
            $tokenModel->where('user_id', $userId)->delete();
            
            // 6. Diğer kullanıcı verilerini temizle (ilanlar, favoriler, vb.)
            $db->table('wdk_listing')
               ->where('user_id', $userId)
               ->update(['is_visible' => 0, 'is_deleted' => 1]);
            
            $db->table('wdk_listing_agent')
               ->where('user_id', $userId)
               ->delete();
            
            $db->table('wdk_favorites')
               ->where('user_id', $userId)
               ->delete();
            
            // İşlemi tamamla
            $db->transComplete();
            
            if ($db->transStatus() === false) {
                throw new \Exception('Veri temizleme işlemi başarısız oldu');
            }
            
            return true;
        } catch (\Exception $e) {
            log_message('error', 'Kullanıcı verileri silinirken hata: ' . $e->getMessage());
            
            // İşlemi geri al
            $db->transRollback();
            
            return false;
        }
    }
}